/*
;;;
;;;; t r a d u t e u r . y
;;;;
;;;; xxl project -- Universite de Nice Sophia Antipolis
;;;;                U.F.R. Sciences - Maitrise d'Informatique
;;;;                project supervisor  V. Granet (vg@unice.fr)
;;;;
;;;; Copyright (C) 1996-1999 U.N.S.A.
;;;; 
;;;; xxl is a free software.  Permission to use, copy, and/or distribute
;;;; this software and its documentation for any purpose and without fee is
;;;; hereby granted, provided that both the above copyright notice and this
;;;; permission notice appear in all copies and derived works. Fees for
;;;; distribution or use of this software or derived works are forbidden.
;;;; This software is provided ``as is'' without any warranty.
;;;; 
;;;;       Author    : Marie Monet
;;;;
;;;; Creation date   : 20-Jul-1997 8:11 
;;;; Last file update: 23-Jul-1999 19:55
;;;;
*/
%{
#include <string.h>
#include <setjmp.h>

#include "stk.h"

#define SYNTAX_ERROR_FUNC "syerror"

typedef enum {RR, RA, AR, AA} celltype;
typedef enum {RELATIVE, ABSOLUTE} coordtype;

int cell_type[2][2] = {{ RR , RA } , { AR, AA }};

#define TYPEOF(c) (cell_type[(c.tr)][(c.tc)])

static Tcl_DString STk_output_string;  

typedef struct {
  coordtype tr, tc;
  int row, col;
} INTERNAL_CELL;

typedef struct {
  char *filename;
  INTERNAL_CELL ci;
} EXTERNAL_CELL;

typedef struct {
  INTERNAL_CELL ci1, ci2;
} INTERNAL_RANGE;

typedef struct {
  char *filename;
  INTERNAL_RANGE ri;
} EXTERNAL_RANGE;

static char *emit_internal_cell(INTERNAL_CELL cell);
static char *emit_external_cell(EXTERNAL_CELL cell);
static char *emit_internal_range(INTERNAL_RANGE range);
static char *emit_external_range(EXTERNAL_RANGE range);
static char *emit_func_call(char *, char *);
static char *emit_op1(char*, char *);
static char *emit_op2(char*, char *, char *);
static char *emit_list(char*, char*);
static void *emit_formula(char*);
%}

%union {
  INTERNAL_CELL  intcell;
  EXTERNAL_CELL  extcell;
  INTERNAL_RANGE intrange;
  EXTERNAL_RANGE extrange;
  char *string;
}	

%left TOK_SUPEG TOK_INFEG TOK_DIFF TOK_EQ '<' '>'
%left TOK_OR
%left TOK_AND
%left '+' '-'
%left '/' '*' '&' '%'
%left UNOP
%token <intcell>  TOK_INTERNAL_CELL
%token <extcell>  TOK_EXTERNAL_CELL
%token <intrange> TOK_INTERNAL_RANGE
%token <extrange> TOK_EXTERNAL_RANGE
%token <string>   TOK_IDENT TOK_NUM TOK_STRING
%token TOK_TRUE TOK_FALSE

%type <string> cell listparam formula expr

%start formula
%%
formula	        : TOK_EQ expr { emit_formula($2); }
                ;

expr            : expr '<' expr { $$ = emit_op2($1,"<",$3); }
                | expr '>' expr { $$ = emit_op2($1,">",$3); }
                | expr TOK_EQ expr { $$ = emit_op2($1,"=",$3); }
                | expr TOK_DIFF expr { $$ = emit_op2($1,"!=",$3); }
                | expr TOK_SUPEG expr { $$ = emit_op2($1,">=",$3); }
                | expr TOK_INFEG expr { $$ = emit_op2($1,"<=",$3); }
                | expr TOK_OR expr { $$ = emit_op2($1,"or",$3); }
                | expr TOK_AND expr { $$ = emit_op2($1,"and",$3); }
                | expr '+' expr { $$ = emit_op2($1,"+",$3); }
                | expr '-' expr { $$ = emit_op2($1,"-",$3); }
                | expr '*' expr { $$ = emit_op2($1,"*",$3); }
                | expr '/' expr { $$ = emit_op2($1,"/",$3); }
                | expr '%' expr { $$ = emit_op2($1,"%",$3); }
                | expr '&' expr { $$ = emit_op2($1,"&",$3); }
                | '-' expr %prec UNOP { $$ = emit_op1("-",$2); }
                | '+' expr %prec UNOP { $$ = emit_op1("+",$2); }
                | '(' expr ')' { $$ = $2; }
                | cell { $$ = $1; }
                | TOK_IDENT '(' listparam ')' { $$ = emit_func_call($1,$3); }
                | TOK_IDENT '(' ')' { $$ = emit_func_call($1,""); }
                | TOK_NUM { $$ = $1; }
                | TOK_STRING { $$ = $1; }
                | TOK_TRUE  { $$ = "#t"; }
                | TOK_FALSE { $$ = "#f"; }
                ;

listparam       : expr               { $$ = $1; }   
                | listparam ',' expr { $$ = emit_list($1,$3); }
                ;

cell            : TOK_INTERNAL_CELL  { $$ = emit_internal_cell($1); }
                | TOK_EXTERNAL_CELL  { $$ = emit_external_cell($1); }
                | TOK_INTERNAL_RANGE { $$ = emit_internal_range($1); }
                | TOK_EXTERNAL_RANGE { $$ = emit_external_range($1); }
                ;
%%

#define MAXBUF 1024

static char *Current_file;
static int Current_row, Current_col;
static char *ext_ref = "#f";

static jmp_buf traducteurEnv;
int yyerror(char *mess) { longjmp(traducteurEnv,1);}

#include "lex.traducteur.c"

static void adjust_bounds(INTERNAL_CELL *c1, INTERNAL_CELL *c2)
{
  int aux;

  if (c1->row > c2-> row) {
    aux = c1->row ; c1->row = c2->row ; c2->row = aux;
    aux = c1->tr ; c1->tr = c2->tr ; c2->tr = aux;
  }

  if (c1->col > c2-> col) {
    aux = c1->col ; c1->col = c2->col ; c2->col = aux;
    aux = c1->tc ; c1->tc = c2->tc ; c2->tc = aux;
  }
}

static void adjust_cell(INTERNAL_CELL *cell)
{
  if (cell->tr == RELATIVE) cell->row -= Current_row;
  if (cell->tc == RELATIVE) cell->col -= Current_col;
}

static char *emit_internal_cell(INTERNAL_CELL cell)
{
  char buf[MAXBUF];

  adjust_cell(&cell);
  sprintf(buf,"#(#f %d (%d . %d))", TYPEOF(cell), cell.row, cell.col);
 
  return strdup(buf);
}

static char *emit_external_cell(EXTERNAL_CELL cell)
{
  char buf[MAXBUF];

  if (strcmp(cell.filename, Current_file) == 0)
    /* internal reference */
    return emit_internal_cell(cell.ci);

  ext_ref = "#t";
  adjust_cell(&(cell.ci));
  sprintf(buf,"#(\"%s\" %d (%d . %d))", cell.filename, 
	  TYPEOF(cell.ci), cell.ci.row, cell.ci.col);
  return strdup(buf);
}

static char *emit_internal_range(INTERNAL_RANGE range)
{
  char buf[MAXBUF];

  adjust_bounds(&(range.ci1),&(range.ci2));
  adjust_cell(&(range.ci1)); adjust_cell(&(range.ci2));
  sprintf(buf,"#(: #f %d (%d . %d) %d (%d . %d))",
	  TYPEOF(range.ci1), range.ci1.row, range.ci1.col,
	  TYPEOF(range.ci2), range.ci2.row, range.ci2.col);

  return strdup(buf);
}

static char *emit_external_range(EXTERNAL_RANGE range)
{
  char buf[MAXBUF];

  if (strcmp(range.filename, Current_file) == 0)
    /* internal reference */
    return emit_internal_range(range.ri);

  ext_ref = "#t";
  adjust_bounds(&(range.ri.ci1),&(range.ri.ci2));
  adjust_cell(&(range.ri.ci1)); adjust_cell(&(range.ri.ci2));
  sprintf(buf,"#(: \"%s\" %d (%d . %d) %d (%d . %d))",
          range.filename,
	  TYPEOF(range.ri.ci1), range.ri.ci1.row, range.ri.ci1.col,
	  TYPEOF(range.ri.ci2), range.ri.ci2.row, range.ri.ci2.col);
  return strdup(buf);
}

static char *emit_op1(char* op, char * a)
{
  char buf[MAXBUF];
  
  sprintf(buf,"(%s %s)",op, a);
  return strdup(buf);
}

static char *emit_op2(char* a, char *op, char *b)
{
  char buf[MAXBUF];
  
  sprintf(buf,"(%s %s %s)",op, a, b);
  return strdup(buf);
}

static char *emit_func_call(char *func, char *expr)
{
  char buf[MAXBUF];
  
  sprintf(buf,"(%s %s)",func, expr);
  return strdup(buf);
}

static char *emit_list(char* l, char* e)
{
  char buf[MAXBUF];
  
  sprintf(buf,"%s %s",l, e);
  return strdup(buf);
}

static void *emit_formula(char* formula)
{
  Tcl_DStringInit(&STk_output_string);
  Tcl_DStringAppend(&STk_output_string, "(cons ", -1);
  Tcl_DStringAppend(&STk_output_string, ext_ref, -1);
  Tcl_DStringAppend(&STk_output_string, " '(", -1);
  Tcl_DStringAppend(&STk_output_string, formula, -1);
  Tcl_DStringAppend(&STk_output_string, "))", -1);
}

/* this function returns a pair :
   car: is a flag indicating any external references
   cdr: is the translation of the string value the cell into
        evaluable sexpr. coord are cell coordinates 
*/

PRIMITIVE STk_traducteur(SCM file, SCM coord, SCM Chaine_Scheme)
{
  /* scan in-memory string input */
  YY_BUFFER_STATE buf_state = traducteur_scan_string(CHARS(Chaine_Scheme));

  Current_file= CHARS(file);
  Current_row = STk_integer_value(CAR(coord));
  Current_col = STk_integer_value(CDR(coord));

  if (setjmp(traducteurEnv) == 0) {
    traducteurparse();
    traducteur_delete_buffer(buf_state);
  }
  else /* syntax error */
    emit_formula(emit_func_call(SYNTAX_ERROR_FUNC,STk_stringify(CHARS(Chaine_Scheme)+1,0))); 
  /* return result */
  return STk_eval_C_string(Tcl_DStringValue(&STk_output_string), NIL);
}

void Xxl_init_traducteur(void) 
{
  STk_add_new_primitive("traducteur", tc_subr_3, STk_traducteur);
}
