/**
;;;
;;;; s y l k 2 x x l . c
;;;;
;;;; xxl project -- Universite de Nice Sophia Antipolis
;;;;                U.F.R. Sciences - Maitrise d'Informatique
;;;;                project supervisor  V. Granet (vg@unice.fr)
;;;;
;;;; Copyright (C) 1996-1999 U.N.S.A.
;;;; 
;;;; xxl is a free software.  Permission to use, copy, and/or distribute
;;;; this software and its documentation for any purpose and without fee is
;;;; hereby granted, provided that both the above copyright notice and this
;;;; permission notice appear in all copies and derived works. Fees for
;;;; distribution or use of this software or derived works are forbidden.
;;;; This software is provided ``as is'' without any warranty.
;;;; 
;;;;       Author    : Marie Monet
;;;;
;;;; Creation date   : 22-Jul-1999 14:10 
;;;; Last file update: 16-Sep-1999 20:43
;;;;
**/

/*
 *  This module defines the primitive "load-sylk" to read and load
 *  a file in sylk format into a xxl spreadsheet
 */

#include <stdlib.h>
#include <stdio.h>
#include <stk.h>
#include "../Lib/utils.h"

extern char *convert_col2a(int col);

#define MAXBUF 1024 /* enough for most values */
#define EOLN   '\n'

static Tcl_DString STk_CurrentCellVal, STk_CurrentCellFormat, STk_FinalRes;

#define  NextField(s)      { \
                               while (*(s) != ';' && *s != EOLN) { \
                                    (s)++; \
                                 } \
                               }


#define  SkipInt(s)      { \
                               while (isdigit(*s) && *s != EOLN) { \
                                    (s)++; \
                                 } \
                               }


#define STk_SETVALUE "(set-table-value! (current-table) %d %d (make <value-cell> \
 :format '(0 . 0) :tag  (make <Table-Tag> :parent (current-table)) :value (celltype \"%s\")))" 

#define STk_SETFORMULA "(let ((f  (traducteur (file-name-of (current-table)) (cons %d %d) \"%s\"))) \
 (set-table-value! (current-table) %d %d (make <formula-cell> \
 :value (eval-formula (current-table) %d %d (cdr f)) \
 :tag  (make <Table-Tag> :parent (current-table)) \
 :format '(0 . 0) :has-ext-ref (car f) :formula (cdr f))))" 

#define  GEN_SET_VALUE(row, col, value) { \
		        sprintf(buf2,STk_SETVALUE,row,col,value); \
                        Tcl_DStringAppend(&STk_CurrentCellVal, buf2, -1); \
                        } 

#define GEN_SET_FORMULA(row, col, formula) { \
		        sprintf(buf2,STk_SETFORMULA,row,col,formula,row, col, row, col); \
                        Tcl_DStringAppend(&STk_CurrentCellVal, buf2, -1); \
                        }

#define NB_FCTS  2 /* To Complete! */

static struct {
  char *sylk_fct, *xxl_fct;
} fct_names_table[NB_FCTS] = {{"AVERAGE", "avg"} , { "PRODUCT", "prod"}};

static void setcellformat(int r, int c, char nf, int precision, char alg)
{
  int xxl_nbf; char buf[64], *xxl_alg;

  switch (nf) {
    case 'D' : xxl_nbf = DEFAULT_F; break;
    case 'F' : xxl_nbf = FIXED; break;
    case 'E' : xxl_nbf = SCIENTIFIC; break;
    case 'C' : xxl_nbf = FINANCIAL; break;
    case '%' : xxl_nbf = PERCENT; break;
    case 'G' : xxl_nbf = DEFAULT_F; break;
    case '$' : xxl_nbf = DOLLAR; break;
    case '*' : xxl_nbf = DEFAULT_F; break;
    default: ;
  }
  Tcl_DStringAppend(&STk_CurrentCellFormat,"(set-format! (get-table-cell (current-table)",-1);
  sprintf(buf, " %d %d) '(%d . %d))", r , c, xxl_nbf, precision);
  Tcl_DStringAppend(&STk_CurrentCellFormat,buf,-1);

   switch (alg) {
   case 'C' : xxl_alg  = "center"; break;
   case 'L' : xxl_alg  = "w"; break;
   case 'R' : xxl_alg  = "e"; break;
   case 'D' : 
   case 'G' : 
   case '-' :
   case 'X' : xxl_alg  = "center";
   }
   Tcl_DStringAppend(&STk_CurrentCellFormat,
     "(let ((t ())) (set! t (make <Table-Tag> :parent (current-table))) (tag-cell t '(",-1);
  sprintf(buf, "%d . %d)) (slot-set! t 'anchor '%s))", r , c, xxl_alg);
  Tcl_DStringAppend(&STk_CurrentCellFormat,buf,-1);
}

/* Produces xxl instructions to set width from cold fcol to lcols
 */
static void setwidth(int fcol, int lcol, int width)
{
  char buf[64]; int i;

  Tcl_DStringAppend(&STk_CurrentCellVal,"(apply width (current-table) '(",-1);
  for (i = fcol; i <= lcol; i++) {
    sprintf(buf, " %d %d", i, width);
    Tcl_DStringAppend(&STk_CurrentCellVal,buf,-1);
  }
  Tcl_DStringAppend(&STk_CurrentCellVal,"))",-1);
}

/*
 *  return from the sylk function name sf 
 *  the corresponding xxl function name
 */
static char* sylkfct2xxlfct(char *sf)
{
  int i;

  for (i = 0; i<NB_FCTS; i++)
    if (strcmp(sf, fct_names_table[i].sylk_fct) == 0)
      return fct_names_table[i].xxl_fct;
  return sf;
}

/* read_ref calculates the col/row value of a sylk reference */
/* It returns 1 for an absolute reference, 0 for a relative reference */
/* and -1 if there is an error */
static int read_ref(char **s, int *n)
{
  int abs = 1, signe = 0;

  if (**s == '(') { /* relative ref */
    abs = 0; (*s)++;
  }

  if (**s == '-' || **s == '+') {
    if (**s == '-') signe = 1;
    (*s)++;
  }
  *n = 0;
  /* **s is a digit */
  if (**s < '0' || **s > '9') /* error */ return -1;
  do {
    *n = *n * 10 + **s - '0';
    (*s)++;
  }
  while (**s >='0' && **s <='9');
  if (! abs) /* there is a ) => skip it*/
    if (**s != ')') /* error */ return -1;
    else  (*s)++;
  if (signe) *n = -*n; 
  /* ok */
  return abs;
}

/* 
 *  Return a xxl expr from the FTD E; field of the RTD C;
 *  
 *  Note: Maybe it should need a really syntax analysis?
 *        Don't like the way this function is programmed
 */
static char *ReadExpr(char **s, int row, int col)
{
  char buf[MAXBUF];
  int i = 1;

  buf[0] = '=';
  while (1) {
    if (**s == EOLN) /* end of field */
      break;
    if (**s == ';')
      if (*((*s)+1) != ';') {
	/* end of field */
	(*s)--;
	break;
      }
      else { /* two semicolons => skip the second one */
	buf[i++] = ';';
	(*s)++;
      }
    else 
      if (**s == '"') { /* scan a string. (A quote is doubled) */ 
	buf[i++] = '\\';  buf[i++] = '"';
	(*s)++;
	while (1) {
	  if (**s == ';') {
	    if (*((*s)+1) != ';') {
	      /* error: should not happend with a correct input */
	      perror("Incorrect E field");
	      break;
	    }
	    else {/* two semicolons => skip the second one */
	      buf[i++] = ';';
	      (*s)++;
	    }
	  }
	  else
	    if (**s == '"') {
	      if (*((*s)+1) == '"') { /* a quote in a string */
		buf[i++]= '\\';  buf[i++]= '\\'; buf[i++]= '\\'; buf[i++]= '"'; 
		(*s)++;
	      }
	      else  { buf[i++] = '\\'; buf[i++] = '"'; break;}
	    }
	    else buf[i++] = **s;
	  (*s)++;
	}
	(*s)++;
      }
      else
	if (isalpha(**s)) { /* a cell reference or a function call */
	  int a1, a2, r , c;
	  char b[MAXBUF], *aux, *spos = *s;

	  if (**s == 'R') { 
	    if (*((*s)+1) == 'C') {
	      /* default row */
	      (*s)++; a1 = r = 0;
	    }
	    else {
	      /* get row number */
	      (*s)++; a1 = read_ref(s, &r);
	      if (a1 == -1) goto funcall;
	    }
	    /* **s should be a C */
	    (*s)++; a2 = read_ref(s, &c);
	    if (a2 == -1) goto funcall;
	    /* ok it was a cell reference */
	    
	    if (a2)  buf[i++] = '$'; else /* relative col ref */ c+=col;
	    aux = convert_col2a(c);
	    /* copy the col value in buf */
	    while (*aux) buf[i++] = *aux++;
	    if (a1)  buf[i++] = '$'; else /* relative row ref */ r+=row;
	    sprintf(b,"%d",r); aux = b;
	    /* copy the row value in buf */
	    while (*aux) buf[i++] = *aux++;
	    continue;
	  }
	  else { /* read the function name */
	    funcall : *s = spos; 
	    r = 0;
	    do
	      b[r++] = *(*s)++;
	    while (isalpha(**s));
	    b[r]='\0';
	    /* translate sylk function name to xxl function name */
	    aux = sylkfct2xxlfct(b);
	    /* copy the function name in buf */
	    while (*aux) buf[i++] = *aux++;
	  }
	}
	else /* any char */
	  buf[i++] = *(*s)++; 
  }
  /* the expression has been read */
  buf[i] = '\0';
  return strdup(buf);
}

/* 
 *  Return the const field of the FTD K; of the RTD C;
 */
static char *ReadKonst(char **s)
{
  char buf[MAXBUF];
  int i = 0;

  if (**s == '"') { /* read a string */
    (*s)++;
    while (1) {
      if (**s == EOLN) /* end of field */
	break;
      if (**s == ';')
	if (*((*s)+1) != ';') {
	  /* end of field */
	  (*s)--;
	  break;
	}
	else { /* two semicolons => skip the second one */
	  buf[i] = ';';
	  (*s)++;
	}
      else  
	if (**s == '"') {
	  buf[i] = '\\'; buf[++i] = '"';
	}
	else /* */
	  buf[i] = **s;
      /* */
      (*s)++; i++; 
    }
    /* remove last \" */
    buf[i-2] = '\0';
    if (strcmp(buf, "true") == 0) return "#t";
    else
      if (strcmp(buf, "false") == 0) return "#f";
    return strdup(buf);
  }
  /* read a number */
  while (**s != ';' && **s != EOLN) {
    buf[i++] = **s;
    (*s)++;
  }
  buf[i] = '\0';
  return strdup(buf);
}

PRIMITIVE STk_load_sylk(SCM filename)
{
  /* filename is a valid sylk file */
  FILE *input = fopen(CHARS(filename), "r");
  char buf1[MAXBUF], buf2[MAXBUF], *ptrBuf1;
  long nbline = 0;
  int maxrow = 0, maxcol = 0, ccol, crow, r, c, wrow, wcol;
  
  Tcl_DStringInit(&STk_CurrentCellVal);   
  Tcl_DStringInit(&STk_CurrentCellFormat);
  while (ptrBuf1 = fgets(buf1, sizeof(buf1), input)) {
    nbline++;
    /* remove blank line */
    if (*ptrBuf1 == EOLN) continue;
    /* *ptrBuf1 is the first char of the sylk line */
    switch (*ptrBuf1) {
      /* ---> RTD = ID (identification) */
    case 'I' :
      if (*++ptrBuf1 == 'D' && *++ptrBuf1 == ';') {
	/* Ok */
      }
      else /* error */ goto error;
      break;
      /* ---> RTD = B (boundary) */
    case 'B' :
      ptrBuf1++;
      while (*ptrBuf1 != EOLN) {
	/* *ptrBuf1 = ';' */
	if (*ptrBuf1++ != ';') goto error;
	switch (*ptrBuf1++) {
	 case 'X' : /* read nb of cols */
	   maxcol = atoi(ptrBuf1);
	   break;
	case 'Y' :  /* read nb of rows */
	  maxrow = atoi(ptrBuf1);
	  break;
	case 'D' :  /* Excel FTD */
	  break;
	default:
	  goto error;
	}
	NextField(ptrBuf1); 
      }
      break;
      /* ---> RTD = C (cell) */
    case 'C' : {
      char *konst = NULL, *expr = NULL;

      ptrBuf1++;
      while (*ptrBuf1 != EOLN) {
	/* *ptrBuf1 == ';' */
	if (*ptrBuf1++ != ';') goto error;
	switch (*ptrBuf1++) {
	case 'X' : /* col coord (diff) */
	  r = read_ref(&ptrBuf1, &c);
	  if (r == -1) goto error;
	  ccol = r ? c : ccol+c;
	  break;
	case 'Y' : /* row coord (diff) */
	  c = read_ref(&ptrBuf1, &r);
	  if (c == -1) goto error;
	  crow = c ? r : crow+r;
	  break;
	case 'K' : /* constant */
	  konst = ReadKonst(&ptrBuf1);
	  break;
	case 'P' : /* cell protected */
	  break;
	case 'E' : /* expression */
	  expr = ReadExpr(&ptrBuf1,crow, ccol);
	  break;
	case 'N' : /* cell not protected */
	  break;
	case 'R' : /* cell row reference */
	  c = read_ref(&ptrBuf1, &r);
	  if (c == -1) goto error;
	  wrow = c ? r : crow+r;
	  break;
	case 'C' : /* cell col reference */
	  r = read_ref(&ptrBuf1, &c);
	  if (r == -1) goto error;
	  wcol = r ? c : ccol+c;
	  break;
	case 'S' : /* shared expr/value given at row ;R and col ;C -- no E; and K; */
	  /* ie (wrow, wcol) */
	  break;
	case 'D' : /* defines shared expression */
	  /* store the expression in a table */
	  break;
	case 'G' : /* defines shared value */ 
	  /* store the value in a table */
	  break;
	default:
	  goto error;
	}
	NextField(ptrBuf1);
      }
      if (expr) { GEN_SET_FORMULA(crow, ccol, expr); }
      else
	if (konst) { GEN_SET_VALUE(crow, ccol, konst); }
      break;
    }
      /* ---> RTD = F (format) */
    case 'F' :      
      { 
	char nbf, alg; 
	int precision, Fftd = 0;

	ptrBuf1++;
	while (*ptrBuf1 != EOLN) {
	  /* *ptrBuf1 == ';' */
	  if (*ptrBuf1++ != ';') goto error;
	  switch (*ptrBuf1++) {
	  case 'X' : /* col coord */
	    r = read_ref(&ptrBuf1, &c);
	    if (r == -1) goto error;
	    ccol = r ? c : ccol+c;
	    break;
	  case 'Y' : /* row coord */
	    c = read_ref(&ptrBuf1, &r);
	    if (c == -1) goto error;
	    crow = c ? r : crow+r;
	    break;
	  case 'R' : /* whole row format */
	    NextField(ptrBuf1); 
	    break;
	  case 'C' : /* whole col format */
	    NextField(ptrBuf1); 
	    break;
	  case 'F' : /* cell formatting properties: ;Fnpa */
	    Fftd = 1; nbf = *ptrBuf1++;
	    precision = atoi(ptrBuf1); SkipInt(ptrBuf1);
	    alg = *ptrBuf1++;
	    NextField(ptrBuf1); 
	    break;
	  case 'D' : /* cell formatting properties with width */
	  case 'K' : /* commas setting */
	  case 'E' : /* formula format option */
	    NextField(ptrBuf1); 
	    break;
	  case 'W' : /* width of columns: ;Wfc lc w*/
	    { 
	      int fc, lc, w;

	      fc = atoi(ptrBuf1);   SkipInt(ptrBuf1); 
	      lc = atoi(++ptrBuf1); SkipInt(ptrBuf1); 
	      w  = atoi(++ptrBuf1);
	      setwidth(fc, lc, w);
	      break;
	    }
	    /* Excel FTDs */
	  case 'N' : /* font to use */
	  case 'P' : /* Excel picture */
	  case 'S' : /* Style */
	  case 'M' : /* ??? */
	  case 'H' : /* Don't show row/column header */
	  case 'G' : /* Don't show gridlines */
	    NextField(ptrBuf1); 
	    break;
	  default:
	    goto error;
	  }
	  NextField(ptrBuf1);  
	}
	if (Fftd)
	  setcellformat(crow, ccol, nbf, precision, alg);
      }
      break;
      /* ---> RTD = P Excel picture format */
    case 'P' :
      break;
      /* ---> RTD = O Excel global option */
    case 'O' :
      break;
      /* ---> RTD = E (end of file) */
    case 'E' :
      break;
    default:
    error: fprintf(stdout,"> error line %d: %s\n",nbline, buf1);
    }
  }
  Tcl_DStringInit(&STk_FinalRes);
  Tcl_DStringAppend(&STk_FinalRes, "(begin ", -1);
  /* set size of the spreadsheet */
  sprintf(buf1, "(extend-sheet (current-table) %d %d)", maxrow, maxcol);
  Tcl_DStringAppend(&STk_FinalRes,buf1, -1);
  Tcl_DStringAppend(&STk_FinalRes,Tcl_DStringValue(&STk_CurrentCellVal),-1);
  Tcl_DStringAppend(&STk_FinalRes,Tcl_DStringValue(&STk_CurrentCellFormat),-1);
  /* reevalute  and close (begin */
  Tcl_DStringAppend(&STk_FinalRes, "(redisplay-all-sheets))", -1); 
  return STk_eval_C_string(Tcl_DStringValue(&STk_FinalRes), NIL); 
}

void Xxl_init_sylk2xxl(void) 
{
  STk_add_new_primitive("load-sylk", tc_subr_1, STk_load_sylk);
}
