/*
;;;
;;;; s y l k . y
;;;;
;;;; xxl project -- Universite de Nice Sophia Antipolis
;;;;                U.F.R. Sciences - Maitrise d'Informatique
;;;;                project supervisor  V. Granet (vg@unice.fr)
;;;;
;;;; Copyright (C) 1996-1999 U.N.S.A.
;;;; 
;;;; xxl is a free software.  Permission to use, copy, and/or distribute
;;;; this software and its documentation for any purpose and without fee is
;;;; hereby granted, provided that both the above copyright notice and this
;;;; permission notice appear in all copies and derived works. Fees for
;;;; distribution or use of this software or derived works are forbidden.
;;;; This software is provided ``as is'' without any warranty.
;;;; 
;;;;       Author    : Marie Monet
;;;;
;;;; Creation date   : 18-Jul-1999 00:25
;;;; Last file update: 23-Jul-1999 19:54
;;;;
*/
%{
#include <string.h>
#include <stdlib.h>

#include "stk.h"

#define PRIO_CMP 0
#define PRIO_ADD 10
#define PRIO_OR  20
#define PRIO_AND 30
#define PRIO_MUL 40
#define PRIO_UNI 50
#define PRIO_MAX 60

#define SYNTAX_ERROR_FUNC "SYERROR"

typedef enum {RR, RA, AR, AA} celltype;

static Tcl_DString STk_output_string;  
static char *emit_cell(char *file, int type, int row, int col);
static char *emit_range(char *file, int type1, int row1, int col1, 
			            int type2, int row2, int col2);
typedef struct {
   char *code;
   int  priority;
} ATTRIBUT;

static void *emit_formula(ATTRIBUT formula);
static char *emit_func_call0(ATTRIBUT func, ATTRIBUT exp);
static char *emit_func_call1(ATTRIBUT func);
static char *emit_exp2(ATTRIBUT op, ATTRIBUT exp1, ATTRIBUT exp2);
static char *emit_exp1(ATTRIBUT op, ATTRIBUT exp1);
static ATTRIBUT make_attr(char *code, int p);
static ATTRIBUT make_string_attr(char *x, int p);
static ATTRIBUT make_list_attr(ATTRIBUT, ATTRIBUT);
static int priority(ATTRIBUT at); 
%}

%union {
  char *string;
  int integer;
  ATTRIBUT at;
}

%token TOK_BEGIN_CELL TOK_BEGIN_RANGE TOK_SUPEG TOK_INFEG TOK_DIFF TOK_EQ TOK_OR TOK_AND
%token <string> TOK_INT  TOK_REAL  TOK_STRING TOK_IDENT
%token TOK_FALSE TOK_TRUE

%type <at> cell expr op func plist
%type <integer> coord
%type <string> file

%start formula

%%
formula	: '(' expr ')' { emit_formula($2); } 
        ;

expr    : '(' op expr expr ')' {$$ =  make_attr(emit_exp2($2, $3, $4),priority($2));}
        | '(' op expr ')'      {$$ =  make_attr(emit_exp1($2, $3),PRIO_UNI); }
        | '(' func plist ')'   {$$ =  make_attr(emit_func_call0($2, $3),priority($2));}
        | '(' func ')'         {$$ =  make_attr(emit_func_call1($2),priority($2));}
        | cell            {$$ = $1;}
        | TOK_INT         {$$ =  make_string_attr($1,PRIO_MAX);}
        | TOK_STRING      {$$ =  make_string_attr($1,PRIO_MAX);}
        | TOK_FALSE       {$$ =  make_string_attr("false",PRIO_MAX);}
        | TOK_TRUE        {$$ =  make_string_attr("true",PRIO_MAX);}
        | TOK_REAL        {$$ =  make_string_attr($1,PRIO_MAX);}
        ;

plist   : expr         {$$ = $1;}
        | plist expr   {$$ = make_list_attr($1, $2); }
        ;

cell    : /* SIMPLE CELL */
          TOK_BEGIN_CELL file TOK_INT '(' coord '.' coord ')' ')'
          { $$ = make_attr(emit_cell($2,atoi($3),$5,$7),PRIO_MAX); }
          /* CELL RANGE*/
        | TOK_BEGIN_RANGE file  TOK_INT '(' coord '.' coord ')' 
          TOK_INT '(' coord '.' coord ')' ')'  	     
          { $$ = make_attr(emit_range($2,atoi($3),$5,$7,
				      atoi($9),$11,$13),PRIO_MAX); }
        ;

file       : TOK_STRING { $$ = $1; }
           | TOK_FALSE  { $$ = "#f"; }
           ; 
coord      : TOK_INT { $$ = atoi($1); }
           | '-' TOK_INT { $$ = -atoi($2); }
           ; 

op         : TOK_EQ    { $$ = make_attr("=",  PRIO_CMP); }
           | TOK_SUPEG { $$ = make_attr(">=", PRIO_CMP); }
           | TOK_INFEG { $$ = make_attr("<=", PRIO_CMP); }
           | TOK_DIFF  { $$ = make_attr("!=", PRIO_CMP); }
           | '<'       { $$ = make_attr("<", PRIO_CMP); }
           | '>'       { $$ = make_attr(">", PRIO_CMP); }
           | TOK_OR    { $$ = make_attr(" or ",PRIO_OR);  }
           | TOK_AND   { $$ = make_attr(" and ",PRIO_AND); }
           | '+'       { $$ = make_attr("+", PRIO_ADD); }
           | '-'       { $$ = make_attr("-", PRIO_ADD); }
           | '/'       { $$ = make_attr("/", PRIO_MUL); }
           | '*'       { $$ = make_attr("*", PRIO_MUL); }
           | '%'       { $$ = make_attr("%", PRIO_MUL); }
           | '&'       { $$ = make_attr("&", PRIO_MUL); }
           ;

func       : TOK_IDENT  { $$ = make_attr($1,PRIO_MAX); }
           ;

%%
extern char *STk_convert_for_Tcl(SCM obj, SCM *res);

#define MAXBUF 1024

static int Current_row, Current_col;

int yyerror(char *mess)
{}

#include "lex.sylk.c"

static char *convert_col2a(int col)
{
 char buf[MAXBUF], *res;
 int i = 0, j = 0;

 /* get all letters */
 do {
   if (col % 26 == 0) {
     buf[i++] = 'Z';
     col = (col / 26) - 1;
   }
   else {
     buf[i++] = col % 26 + 'A' - 1;
     col /= 26;
   }
 } while (col != 0);

 /* construct  the result */
 res = (char *) malloc((i + 1)); 

 while (i>0) res[j++] = buf[--i];
 res[j] = '\0';

 return res;
}

static ATTRIBUT make_attr(char *code, int p)
{
  ATTRIBUT at;

  at.priority = p;
  at.code = code;
  return at;
}

static ATTRIBUT make_string_attr(char *x, int p)
{
  ATTRIBUT at;
  char buf[MAXBUF];

  at.priority = p;
  sprintf(buf, "%s",x);
  at.code = strdup(buf);
  return at;
}

static ATTRIBUT make_list_attr(ATTRIBUT l, ATTRIBUT e)
{
  ATTRIBUT at;
  char buf[MAXBUF];

  at.priority = e.priority;
  sprintf(buf,"%s,%s",l.code,e.code);
  at.code = strdup(buf);
  return at;
}

static int priority(ATTRIBUT at)
{
  return at.priority;
}

static char *make_coord(int type, int row, int col)
{
  char buf[MAXBUF];
  
  switch (type) { 
  case RR: 
    sprintf(buf,"R(%+d)C(%+d)", row, col);
    break;
  case AR: 
    sprintf(buf,"R%dC(%+d)", row, col);
    break;
  case RA:
    sprintf(buf,"R(%+d)C%d", row, col);
	break;
  case AA:
    sprintf(buf,"R%dC%d", row, col);
  }
  return strdup(buf);
}

static char *emit_cell(char *file, int type, int row, int col)
{
  char buf[MAXBUF];

  if (strcmp(file,"#f") == 0)
    return make_coord(type,row,col);
  else {
    sprintf(buf,"%s%s",file,make_coord(type,row,col));
    return strdup(buf);
  }
}

static char *emit_range(char *file, int type1, int row1, int col1, 
			            int type2, int row2, int col2)
{
  char buf[MAXBUF];
  char *coord1, *coord2;

  if (strcmp(file,"#f") == 0)
    sprintf(buf,"%s:%s",
	    make_coord(type1,row1,col1),
	    make_coord(type2,row2,col2));
  else
    sprintf(buf,"%s%s:%s",file,
	    make_coord(type1,row1,col1),
	    make_coord(type2,row2,col2));
  
  return strdup(buf);
}

static char *emit_exp1(ATTRIBUT op, ATTRIBUT exp1)
{
  char buf[MAXBUF];

  sprintf(buf,"%s%s",op.code, exp1.code);
  return strdup(buf);
}

static char *emit_exp2(ATTRIBUT op, ATTRIBUT exp1, ATTRIBUT exp2)
{
  char buf1[MAXBUF], buf2[MAXBUF];

  if (op.priority > exp1.priority)
    sprintf(buf1,"(%s)%s",exp1.code,op.code);
  else 
    sprintf(buf1,"%s%s",exp1.code,op.code);

  if (op.priority > exp2.priority)
    sprintf(buf2,"%s(%s)",buf1,exp2.code);
  else 
    sprintf(buf2,"%s%s",buf1,exp2.code);

  return strdup(buf2);
}

static char *replace_quote(char *s)
/* remove from string s : first and and last quote */
/* and replace all \" by " */
/* FIX: replace also all \\...\\" by \" */
{ 
  register int i = 0, j = 1;
  char buf[MAXBUF];

  while (s[j] != '\0') {
    if (s[j] == '\\' && s[j+1] == '"') j++;
    buf[i++] = s[j++]; 
  }
  /* remove last quote */
  buf[i-1] = '\0';
  return strdup(buf);
}

static char *emit_func_call0(ATTRIBUT func, ATTRIBUT exp)
     /* emit code for a function call with arguments.
	Special treatement for syerror which is the 
	function of syntax error deteted by 
	<< traducteur >> parser.
     */
{
  char buf[MAXBUF];

  if (strcmp(func.code, SYNTAX_ERROR_FUNC) == 0)
    /* syerror function means an syntax error detected
       by traducteur. String value of exp.code is formula
       input with the syntax error
    */
    sprintf(buf,"%s", replace_quote(exp.code));
  else
    /* a "normal" function */
    sprintf(buf,"%s(%s)",func.code, exp.code);
  return strdup(buf);
}

static char *emit_func_call1(ATTRIBUT func)
     /* function without arguments */
{
  char buf[MAXBUF];
  sprintf(buf,"%s()",func.code);
  return strdup(buf);
}

static void *emit_formula(ATTRIBUT formula)
{
  Tcl_DStringAppend(&STk_output_string, formula.code, -1);
}

PRIMITIVE STk_sylk(SCM row, SCM col, SCM List_Scheme)
{
  SCM res; /* not used */
  /* scan in-memory string input */
  YY_BUFFER_STATE buf_state = 
    sylk_scan_string(STk_convert_for_Tcl(List_Scheme,&res));

  Current_row = STk_integer_value(row);
  Current_col = STk_integer_value(col);
  Tcl_DStringInit(&STk_output_string);
  sylkparse();
  sylk_delete_buffer(buf_state);
  /*   return STk_eval_C_string(STk_stringify(Tcl_DStringValue(&STk_output_string),0)
       ,NIL); */
 return  STk_makestring(Tcl_DStringValue(&STk_output_string));
}

void Xxl_init_sylk(void) 
{
  STk_add_new_primitive("sylk", tc_subr_3, STk_sylk);
}
